import { NextFunction, Request, Response } from 'express';
import { MitraRequestService } from '../service/mitra-request-service';
import {
  CreateMitraRequestRequest,
  MitraRequestQueryParams,
  UpdateMitraRequestStatusRequest,
} from '../model/mitra-request-model';
import { UserRequest } from '../type/user-request';

export class MitraRequestController {
  static async create(req: UserRequest, res: Response, next: NextFunction) {
    try {
      // Extract form data and files
      const files = req.files as { [fieldname: string]: Express.Multer.File[] };

      // Ensure that both ktp and payment_proof are uploaded
      if (!files.ktp || !files.payment) {
        return res.status(400).json({
          success: false,
          message: 'KTP dan bukti pembayaran harus diunggah',
        });
      }

      // Extract paths of the uploaded files
      const ktpFilePath = `/public/upload/ktp/${files.ktp[0].filename}`;
      const paymentFilePath = `/public/upload/payment/${files.payment[0].filename}`;

      const request: CreateMitraRequestRequest = {
        ktp: ktpFilePath,
        upgrade_amount: Number(req.body.upgrade_amount),
        payment_proof: paymentFilePath,
      };
      const response = await MitraRequestService.create(req.user, request);

      res.status(201).json({
        success: true,
        message: 'Mitra request created successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async getStatusByUser(req: UserRequest, res: Response, next: NextFunction) {
    try {
      const response = await MitraRequestService.getStatusByUser(req.user);

      res.status(200).json({
        success: true,
        message: 'Last mitra request status retrieved successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async getAll(req: Request, res: Response, next: NextFunction) {
    try {
      const queryParams: MitraRequestQueryParams = {
        search: req.query.search as string,
        status: req.query.status as 'PENDING' | 'APPROVED' | 'REJECTED',
        sort: req.query.sort as string,
        order: req.query.order as 'asc' | 'desc',
        page: Number(req.query.page),
        limit: Number(req.query.limit),
      };
      const response = await MitraRequestService.getAll(queryParams);

      res.status(200).json({
        success: true,
        message: 'Mitra requests retrieved successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async get(req: Request, res: Response, next: NextFunction) {
    try {
      const id = req.params.id;
      const response = await MitraRequestService.get(id);

      res.status(200).json({
        success: true,
        message: 'Mitra request retrieved successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async updateStatus(req: Request, res: Response, next: NextFunction) {
    try {
      const id = req.params.id;
      const statusRequest: UpdateMitraRequestStatusRequest = req.body as UpdateMitraRequestStatusRequest;
      await MitraRequestService.updateStatus(id, statusRequest);

      res.status(200).json({
        success: true,
        message: 'Mitra request status updated successfully',
      });
    } catch (error) {
      next(error);
    }
  }

  static async delete(req: Request, res: Response, next: NextFunction) {
    try {
      const id = req.params.id;
      await MitraRequestService.delete(id);

      res.status(200).json({
        success: true,
        message: 'Mitra request deleted successfully',
      });
    } catch (error) {
      next(error);
    }
  }
}
