import { NextFunction, Request, Response } from 'express';
import { ContentService } from '../service/content-service';
import { CreateContentRequest, UpdateContentRequest, UpdateContentValueRequest, CreateColorRequest, UpdateColorRequest } from '../model/content-model';

export class ContentController {
  static async create(req: Request, res: Response, next: NextFunction) {
    try {
      const request: CreateContentRequest = req.body as CreateContentRequest;
      const response = await ContentService.create(request);

      res.status(201).json({
        success: true,
        message: 'Content created successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async update(req: Request, res: Response, next: NextFunction) {
    try {
      let imagePath: string | undefined;
      if (req.file) {
        imagePath = `/public/content/${req.file.filename}`;
      }

      const id = req.params.id;
      const request: UpdateContentRequest = {
        page: req.body.page,
        section: req.body.section,
        type: req.body.type,
        key: req.body.key,
        value: imagePath || req.body.value,
      };

      const response = await ContentService.update(id, request);

      res.status(200).json({
        success: true,
        message: 'Content updated successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async updateValue(req: Request, res: Response, next: NextFunction) {
    try {
      const id = req.params.id;
      const request: UpdateContentValueRequest = {
        value: req.body.value ? JSON.parse(req.body.value) : '',
      };
      const response = await ContentService.updateValue(id, request, req.file);

      res.status(200).json({
        success: true,
        message: 'Content updated successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async updateVisibility(req: Request, res: Response, next: NextFunction) {
    try {
      const id = req.params.id;
      const visible = req.body.visible;
      const response = await ContentService.updateVisibility(id, visible);

      res.status(200).json({
        success: true,
        message: 'Content visibility updated successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async delete(req: Request, res: Response, next: NextFunction) {
    try {
      const id = req.params.id;
      await ContentService.delete(id);

      res.status(200).json({
        success: true,
        message: 'Content deleted successfully',
      });
    } catch (error) {
      next(error);
    }
  }

  static async getAll(req: Request, res: Response, next: NextFunction) {
    try {
      const response = await ContentService.getAll();

      res.status(200).json({
        success: true,
        message: 'Contents retrieved successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async getAllByPage(req: Request, res: Response, next: NextFunction) {
    try {
      const page = req.params.page;
      const response = await ContentService.getAllByPage(page);

      res.status(200).json({
        success: true,
        message: 'Contents retrieved successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async getAllBySection(req: Request, res: Response, next: NextFunction) {
    try {
      const section = req.params.section;
      const response = await ContentService.getAllBySection(section);

      res.status(200).json({
        success: true,
        message: 'Contents retrieved successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async get(req: Request, res: Response, next: NextFunction) {
    try {
      const id = req.params.id;
      const response = await ContentService.get(id);

      res.status(200).json({
        success: true,
        message: 'Content retrieved successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async getAllColors(req: Request, res: Response, next: NextFunction) {
    try {
      const response = await ContentService.getAllColors();

      res.status(200).json({
        success: true,
        message: 'Colors retrieved successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async getColor(req: Request, res: Response, next: NextFunction) {
    try {
      const id = req.params.id;
      const response = await ContentService.getColor(id);

      res.status(200).json({
        success: true,
        message: 'Color retrieved successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async createColor(req: Request, res: Response, next: NextFunction) {
    try {
      const request: CreateColorRequest = req.body as CreateColorRequest;
      const response = await ContentService.createColor(request);

      res.status(201).json({
        success: true,
        message: 'Color created successfully',
        data: response,
      });
    } catch (error) {
      next(error);
    }
  }

  static async updateColors(req: Request, res: Response, next: NextFunction) {
    try {
      const request: UpdateColorRequest[] = req.body as UpdateColorRequest[];
      await ContentService.updateColors(request);

      res.status(200).json({
        success: true,
        message: 'Color updated successfully',
      });
    } catch (error) {
      next(error);
    }
  }

  static async deleteColor(req: Request, res: Response, next: NextFunction) {
    try {
      const id = req.params.id;
      await ContentService.deleteColor(id);

      res.status(200).json({
        success: true,
        message: 'Color deleted successfully',
      });
    } catch (error) {
      next(error);
    }
  }
}
