import * as assert from 'assert';
import test, { describe } from 'node:test';
import parse from '../parse.js';
import stringify from '../stringify.js';
import v7, { updateV7State } from '../v7.js';
const RFC_V7 = '017f22e2-79b0-7cc3-98c4-dc0c0c07398f';
const RFC_V7_BYTES = parse('017f22e2-79b0-7cc3-98c4-dc0c0c07398f');
const RFC_MSECS = 0x17f22e279b0;
const RFC_SEQ = (0x0cc3 << 20) | (0x98c4dc >> 2);
const RFC_RANDOM = Uint8Array.of(0x10, 0x91, 0x56, 0xbe, 0xc4, 0xfb, 0x0c, 0xc3, 0x18, 0xc4, 0x6c, 0x0c, 0x0c, 0x07, 0x39, 0x8f);
describe('v7', () => {
    test('subsequent UUIDs are different', () => {
        const id1 = v7();
        const id2 = v7();
        assert.ok(id1 !== id2);
    });
    test('explicit options.random and options.msecs produces expected result', () => {
        const id = v7({
            random: RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        });
        assert.strictEqual(id, RFC_V7);
    });
    test('explicit options.rng produces expected result', () => {
        const id = v7({
            rng: () => RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        });
        assert.strictEqual(id, RFC_V7);
    });
    test('explicit options.msecs produces expected result', () => {
        const id = v7({
            msecs: RFC_MSECS,
        });
        assert.strictEqual(id.indexOf('017f22e2'), 0);
    });
    test('fills one UUID into a buffer as expected', () => {
        const buffer = new Uint8Array(16);
        const result = v7({
            random: RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        }, buffer);
        stringify(buffer);
        assert.deepEqual(buffer, RFC_V7_BYTES);
        assert.strictEqual(buffer, result);
    });
    test('fills two UUIDs into a buffer as expected', () => {
        const buffer = new Uint8Array(32);
        v7({
            random: RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        }, buffer, 0);
        v7({
            random: RFC_RANDOM,
            msecs: RFC_MSECS,
            seq: RFC_SEQ,
        }, buffer, 16);
        const expected = new Uint8Array(32);
        expected.set(RFC_V7_BYTES);
        expected.set(RFC_V7_BYTES, 16);
        assert.deepEqual(buffer, expected);
    });
    test('lexicographical sorting is preserved', () => {
        let id;
        let prior;
        let msecs = RFC_MSECS;
        for (let i = 0; i < 20000; ++i) {
            if (i % 1500 === 0) {
                msecs += 1;
            }
            id = v7({ msecs, seq: i });
            if (prior !== undefined) {
                assert.ok(prior < id, `${prior} < ${id}`);
            }
            prior = id;
        }
    });
    test('can supply seq', () => {
        let seq = 0x12345;
        let uuid = v7({
            msecs: RFC_MSECS,
            seq,
        });
        assert.strictEqual(uuid.substr(0, 25), '017f22e2-79b0-7000-848d-1');
        seq = 0x6fffffff;
        uuid = v7({
            msecs: RFC_MSECS,
            seq,
        });
        assert.strictEqual(uuid.substring(0, 25), '017f22e2-79b0-76ff-bfff-f');
    });
    test('internal seq is reset upon timestamp change', () => {
        v7({
            msecs: RFC_MSECS,
            seq: 0x6fffffff,
        });
        const uuid = v7({
            msecs: RFC_MSECS + 1,
        });
        assert.ok(uuid.indexOf('fff') !== 15);
    });
    test('v7() state transitions', () => {
        const tests = [
            {
                title: 'new time interval',
                state: { msecs: 1, seq: 123 },
                now: 2,
                expected: {
                    msecs: 2,
                    seq: 0x6c318c4,
                },
            },
            {
                title: 'same time interval',
                state: { msecs: 1, seq: 123 },
                now: 1,
                expected: {
                    msecs: 1,
                    seq: 124,
                },
            },
            {
                title: 'same time interval (sequence rollover)',
                state: { msecs: 1, seq: 0xffffffff },
                now: 1,
                expected: {
                    msecs: 2,
                    seq: 0,
                },
            },
            {
                title: 'time regression',
                state: { msecs: 2, seq: 123 },
                now: 1,
                expected: {
                    msecs: 2,
                    seq: 124,
                },
            },
            {
                title: 'time regression (sequence rollover)',
                state: { msecs: 2, seq: 0xffffffff },
                now: 1,
                expected: {
                    msecs: 3,
                    seq: 0,
                },
            },
        ];
        for (const { title, state, now, expected } of tests) {
            assert.deepStrictEqual(updateV7State(state, now, RFC_RANDOM), expected, `Failed: ${title}`);
        }
    });
    test('flipping bits changes the result', () => {
        const asBigInt = (buf) => buf.reduce((acc, v) => (acc << 8n) | BigInt(v), 0n);
        const asNumber = (bits, data) => Number(BigInt.asUintN(bits, data));
        const flip = (data, n) => data ^ (1n << BigInt(127 - n));
        const optionsFrom = (data) => {
            const ms = asNumber(48, data >> 80n);
            const hi = asNumber(12, data >> 64n);
            const lo = asNumber(20, data >> 42n);
            const r = BigInt.asUintN(42, data);
            return {
                msecs: ms,
                seq: (hi << 20) | lo,
                random: Uint8Array.from([
                    ...Array(10).fill(0),
                    ...Array(6)
                        .fill(0)
                        .map((_, i) => asNumber(8, r >> (BigInt(i) * 8n)))
                        .reverse(),
                ]),
            };
        };
        const buf = new Uint8Array(16);
        const data = asBigInt(v7({}, buf));
        const id = stringify(buf);
        const reserved = [48, 49, 50, 51, 64, 65];
        for (let i = 0; i < 128; ++i) {
            if (reserved.includes(i)) {
                continue;
            }
            const flipped = flip(data, i);
            assert.strictEqual(asBigInt(v7(optionsFrom(flipped), buf)).toString(16), flipped.toString(16), `Unequal uuids at bit ${i}`);
            assert.notStrictEqual(stringify(buf), id);
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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