"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.db = void 0;
const promise_1 = __importDefault(require("mysql2/promise"));
const logger_1 = require("./logger");
const dotenv_1 = require("dotenv");
(0, dotenv_1.config)();
class Database {
    constructor() {
        this.pool = promise_1.default.createPool({
            host: process.env.DB_HOST,
            user: process.env.DB_USER,
            password: process.env.DB_PASSWORD,
            database: process.env.DB_NAME,
            waitForConnections: true,
            connectionLimit: 10,
            queueLimit: 0,
        });
    }
    // Execute a query, automatically handles connections and errors
    query(query_1) {
        return __awaiter(this, arguments, void 0, function* (query, params = []) {
            let connection;
            const start = Date.now(); // Waktu mulai eksekusi query
            try {
                connection = yield this.pool.getConnection();
                const [rows] = yield connection.query(query, params);
                const duration = Date.now() - start; // Menghitung waktu eksekusi
                logger_1.logger.info('Query executed', { query, params, duration: `${duration} ms` });
                return rows;
            }
            catch (error) {
                logger_1.logger.error('Database Query Error', { error });
                throw new Error('Failed to execute query');
            }
            finally {
                if (connection)
                    connection.release();
            }
        });
    }
    // Execute a query but only return a single row or null if no rows found
    queryOne(query_1) {
        return __awaiter(this, arguments, void 0, function* (query, params = []) {
            const rows = yield this.query(query, params);
            return rows.length > 0 ? rows[0] : null;
        });
    }
    // Begin a transaction, allowing multiple queries in one transaction
    transaction(callback) {
        return __awaiter(this, void 0, void 0, function* () {
            const connection = yield this.pool.getConnection();
            yield connection.beginTransaction();
            try {
                const result = yield callback(connection);
                yield connection.commit();
                return result;
            }
            catch (error) {
                yield connection.rollback();
                logger_1.logger.error('Transaction failed', { error });
                throw new Error('Transaction failed');
            }
            finally {
                connection.release();
            }
        });
    }
    // Clean up connections (for testing or app shutdown)
    close() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.pool.end();
        });
    }
}
exports.db = new Database();
process.on('SIGINT', () => __awaiter(void 0, void 0, void 0, function* () {
    try {
        yield exports.db.close();
        logger_1.logger.info('Database connections closed gracefully');
    }
    catch (err) {
        logger_1.logger.error('Error closing database connections', err);
    }
    finally {
        process.exit();
    }
}));
