/*
 * This is a script for seeding the database with initial data.
 * Run this script using `npm run seed`
 */

import { db } from '../src/application/database';
import { v4 as uuidv4 } from 'uuid';
import bcrypt from 'bcrypt';

type Role = 'ADMIN' | 'USER' | 'MITRA';
const users = [
  {
    email: 'admin@example.com',
    full_name: 'Admin',
    whatsapp_number: '1234567890',
    password: 'admin123',
    role: 'ADMIN',
  },
  {
    email: 'user@example.com',
    full_name: 'User',
    whatsapp_number: '1234567890',
    password: 'user123',
    role: 'USER',
  },
  {
    email: 'mitra@example.com',
    full_name: 'Mitra',
    whatsapp_number: '1234567890',
    password: 'mitra123',
    role: 'MITRA',
  },
];

const packageTypes = [
  { name: 'Tiket Pesawat' },
  { name: 'Hotel Makkah' },
  { name: 'Hotel Madinah' },
  { name: 'Transportasi' },
  { name: 'Muthawif' },
  { name: 'Handling' },
];

type PackageTypes = 'TiketPesawat' | 'HotelMakkah' | 'HotelMadinah' | 'Transportasi' | 'Muthawif' | 'Handling';
const itemData: Record<PackageTypes, Array<{ name: string; price: number; order_number?: number }>> = {
  TiketPesawat: [
    { name: 'Tanpa Tiket Pesawat', price: 0, order_number: 1 },
    { name: 'Saudia Economy Class - CGK-JED', price: 5000000, order_number: 2 },
    { name: 'Saudia Economy Class - CGK-MED', price: 10000000, order_number: 3 },
    { name: 'Etihat Economy Class - CGK-JED', price: 5000000, order_number: 4 },
    { name: 'Indigo Economy Class - CGK-JED', price: 10000000, order_number: 5 },
    { name: 'Garuda Economy Class - CGK-JED', price: 15000000, order_number: 6 },
    { name: 'Garuda Economy Class - CGK-MED', price: 15000000, order_number: 7 },
  ],
  HotelMakkah: [
    { name: 'Tanpa Hotel', price: 0, order_number: 1 },
    { name: 'Hotel Bintang 3', price: 1000000, order_number: 2 },
    { name: 'Hotel Bintang 4', price: 1500000, order_number: 3 },
    { name: 'Hotel Bintang 5', price: 2000000, order_number: 4 },
  ],
  HotelMadinah: [
    { name: 'Tanpa Hotel', price: 0, order_number: 1 },
    { name: 'Hotel Bintang 3', price: 800000, order_number: 2 },
    { name: 'Hotel Bintang 4', price: 1200000, order_number: 3 },
    { name: 'Hotel Bintang 5', price: 1600000, order_number: 4 },
  ],
  Transportasi: [
    { name: 'Tanpa Transportasi', price: 0, order_number: 1 },
    { name: 'Dengan Transportasi', price: 500000, order_number: 2 },
  ],
  Muthawif: [
    { name: 'Tanpa Muthawwif', price: 0, order_number: 1 },
    { name: 'Muthawwif 4D', price: 500000, order_number: 2 },
    { name: 'Full Muthawwif', price: 1000000, order_number: 3 },
  ],
  Handling: [
    { name: 'Tanpa Handling', price: 0, order_number: 1 },
    { name: 'Handling', price: 500000, order_number: 2 },
    { name: 'Full Handling', price: 1000000, order_number: 3 },
  ],
};

type PeriodCategory = 'LOW' | 'MID' | 'HIGH' | 'RAMADHAN' | 'ITIKAF';
const CategoryData = [{ name: 'Paket Umroh' }, { name: 'Visa Umroh' }, { name: 'Paket Hotel' }];

const periods = [
  { category: 'LOW', start_date: new Date('2024-09-01'), end_date: new Date('2024-11-30') },
  { category: 'MID', start_date: new Date('2024-12-01'), end_date: new Date('2024-12-16') },
  { category: 'HIGH', start_date: new Date('2024-12-17'), end_date: new Date('2025-01-11') },
  { category: 'MID', start_date: new Date('2025-01-12'), end_date: new Date('2025-02-28') },
  { category: 'RAMADHAN', start_date: new Date('2025-03-01'), end_date: new Date('2025-03-15') },
  { category: 'ITIKAF', start_date: new Date('2025-03-16'), end_date: new Date('2025-03-31') },
  { category: 'MID', start_date: new Date('2025-04-01'), end_date: new Date('2025-04-10') },
  { category: 'LOW', start_date: new Date('2025-04-11'), end_date: new Date('2025-04-30') },
];

const vendors = [{ name: 'MKM' }, { name: 'ALH' }, { name: 'HVN' }, { name: 'ALK' }];

type City = 'MEKKAH' | 'MADINAH';
const hotels = [
  {
    vendor: 'MKM',
    hotels: [
      { name: 'TIDAK', city: 'MEKKAH', prices: { double: 0, triple: 0, quad: 0 } },
      { name: 'AKABER AL HIJRAH', city: 'MEKKAH', prices: { double: 250, triple: 290, quad: 340 } },
      { name: 'SNOOD AJYAD', city: 'MEKKAH', prices: { double: 420, triple: 460, quad: 500 } },
      { name: 'GRAND AL MASSA', city: 'MEKKAH', prices: { double: 490, triple: 540, quad: 590 } },
      { name: 'RAYYANA AJYAD', city: 'MEKKAH', prices: { double: 660, triple: 670, quad: 770 } },
      { name: 'MARRIOT JABAL OMAR', city: 'MEKKAH', prices: { double: 660, triple: 760, quad: 860 } },
      { name: 'MAKAREM AJYAD', city: 'MEKKAH', prices: { double: 660, triple: 790, quad: 920 } },
      { name: 'SOFWAH DURRAR AL EIMAN', city: 'MEKKAH', prices: { double: 980, triple: 1160, quad: 1325 } },
      { name: 'PULLMAN ZAM ZAM', city: 'MEKKAH', prices: { double: 1080, triple: 1225, quad: 1450 } },
      { name: 'HYAT REGENCY MAKKAH', city: 'MEKKAH', prices: { double: 1150, triple: 1450, quad: 1750 } },
      { name: 'TIDAK', city: 'MADINAH', prices: { double: 0, triple: 0, quad: 0 } },
      { name: 'SAMA AL MASI', city: 'MADINAH', prices: { double: 390, triple: 430, quad: 470 } },
      { name: 'JAWHARAT RASHEED', city: 'MADINAH', prices: { double: 360, triple: 400, quad: 440 } },
      { name: 'HAYAH PLAZA', city: 'MADINAH', prices: { double: 380, triple: 420, quad: 460 } },
      { name: 'AL MADINAH CONCORDE', city: 'MADINAH', prices: { double: 425, triple: 470, quad: 515 } },
      { name: 'RUA INTERNATIONAL', city: 'MADINAH', prices: { double: 430, triple: 475, quad: 520 } },
      { name: 'DAR AL NAEM', city: 'MADINAH', prices: { double: 445, triple: 490, quad: 535 } },
      { name: 'SAFWAT AL MADINAH', city: 'MADINAH', prices: { double: 480, triple: 525, quad: 570 } },
      { name: 'GRAND PLAZA', city: 'MADINAH', prices: { double: 490, triple: 560, quad: 620 } },
      { name: 'FRONT TAIBA', city: 'MADINAH', prices: { double: 720, triple: 845, quad: 970 } },
      { name: 'AL AQEEQ', city: 'MADINAH', prices: { double: 750, triple: 870, quad: 990 } },
      { name: 'AL HARAM DAR ELIMAN', city: 'MADINAH', prices: { double: 830, triple: 955, quad: 1180 } },
      { name: 'HILTON MADINAH', city: 'MADINAH', prices: { double: 1709, triple: 2210, quad: 2550 } },
    ],
  },
  {
    vendor: 'ALH',
    hotels: [
      { name: 'SARAYA ALTAJ', city: 'MEKKAH', prices: { double: 240, triple: 255, quad: 290 } },
      { name: 'AMJAD AJYAD', city: 'MEKKAH', prices: { double: 220, triple: 260, quad: 300 } },
      { name: 'RAWABI ZAM ZAM', city: 'MEKKAH', prices: { double: 280, triple: 315, quad: 350 } },
      { name: 'PARK REGIS', city: 'MEKKAH', prices: { double: 260, triple: 310, quad: 360 } },
      { name: 'MATHER AL JEWAR', city: 'MEKKAH', prices: { double: 285, triple: 325, quad: 365 } },
      { name: 'AL MASSA BADR', city: 'MEKKAH', prices: { double: 305, triple: 345, quad: 385 } },
      { name: 'REHAB AL TAQWA', city: 'MEKKAH', prices: { double: 305, triple: 345, quad: 385 } },
      { name: 'MILLENIUM HOTEL', city: 'MEKKAH', prices: { double: 320, triple: 380, quad: 440 } },
      { name: 'VOCO MAKKAH', city: 'MEKKAH', prices: { double: 330, triple: 385, quad: 440 } },
      { name: 'LE MERIDIEN TOWER', city: 'MEKKAH', prices: { double: 350, triple: 400, quad: 460 } },
      { name: 'ZILAL AL NUZULA', city: 'MEKKAH', prices: { double: 360, triple: 405, quad: 450 } },
      { name: 'AJYAD ALJIWAR', city: 'MEKKAH', prices: { double: 450, triple: 485, quad: 520 } },
      { name: 'DANA ALTAJ', city: 'MEKKAH', prices: { double: 450, triple: 485, quad: 520 } },
      { name: 'JAWAD ALTAJ', city: 'MEKKAH', prices: { double: 450, triple: 485, quad: 520 } },
      { name: 'ELAF AJYAD', city: 'MEKKAH', prices: { double: 450, triple: 495, quad: 540 } },
      { name: 'SNOOD AJYAD', city: 'MEKKAH', prices: { double: 460, triple: 500, quad: 540 } },
      { name: 'FOUR POINT BY SHERATON', city: 'MEKKAH', prices: { double: 290, triple: 410, quad: 550 } },
      { name: 'RAMADA DAR AL FAIZEEN', city: 'MEKKAH', prices: { double: 475, triple: 525, quad: 575 } },
      { name: 'FAJR BADIE 2 (MAYSAN)', city: 'MEKKAH', prices: { double: 490, triple: 540, quad: 590 } },
      { name: 'GRAND AL MASSA', city: 'MEKKAH', prices: { double: 545, triple: 595, quad: 645 } },
      { name: 'MIRA AJYAD', city: 'MEKKAH', prices: { double: 530, triple: 590, quad: 650 } },
      { name: 'AZKA AL SAFA', city: 'MEKKAH', prices: { double: 570, triple: 640, quad: 710 } },
      { name: 'OLAYAN AL HARAM', city: 'MEKKAH', prices: { double: 530, triple: 645, quad: 730 } },
      { name: 'AL SHOHADA', city: 'MEKKAH', prices: { double: 630, triple: 730, quad: 830 } },
      { name: 'HILTON CONVENTION', city: 'MEKKAH', prices: { double: 890, triple: 1060, quad: 1230 } },
      { name: 'PULLMAN ZAM ZAM', city: 'MEKKAH', prices: { double: 950, triple: 1175, quad: 1400 } },
      { name: 'RAFFLES MAKKAH PALACE', city: 'MEKKAH', prices: { double: 1610, triple: 1760, quad: 2080 } },
      { name: 'JAWHARAT RASHEED', city: 'MADINAH', prices: { double: 360, triple: 400, quad: 440 } },
      { name: 'DAR ELIMAN AL HARAM', city: 'MADINAH', prices: { double: 1000, triple: 1250, quad: 3000 } },
      { name: 'GOLDEN TULIP ZAHABI', city: 'MADINAH', prices: { double: 1065, triple: 1115, quad: 1180 } },
    ],
  },
];

const airlines = [
  { name: 'TIDAK', price: 0 },
  { name: 'GARUDA', price: 15600000 },
  { name: 'SAUDIA', price: 14600000 },
  { name: 'QATAR', price: 14500000 },
  { name: 'ETIHAD', price: 14500000 },
  { name: 'OMAN', price: 13500000 },
  { name: 'SCOOT', price: 13500000 },
  { name: 'LION', price: 12500000 },
  { name: 'INDIGO', price: 11900000 },
  { name: 'AIR ASIA KUL', price: 11500000 },
  { name: '9 JUTA', price: 9000000 },
  { name: '10 JUTA', price: 10000000 },
  { name: '11 JUTA', price: 11000000 },
  { name: '12 JUTA', price: 12000000 },
  { name: '13 JUTA', price: 13000000 },
  { name: '14 JUTA', price: 14000000 },
  { name: '15 JUTA', price: 15000000 },
  { name: '16 JUTA', price: 16000000 },
  { name: '17 JUTA', price: 17000000 },
  { name: 'SV JED JED DEC', price: 16400000 },
  { name: 'SV MED JED DEC', price: 17400000 },
];

const visa = [
  { name: 'TIDAK', price_idr: 0 },
  { name: 'VISA ONLY', price_idr: 2106000 },
  { name: 'VISA & BUS (35)', price_idr: 2184000 },
  { name: 'VISA & BUS THAIF (35)', price_idr: 2324400 },
  { name: 'VISA BUS 1', price_idr: 15444000 },
  { name: 'VISA BUS 2', price_idr: 8626800 },
  { name: 'VISA BUS 3', price_idr: 6349200 },
  { name: 'VISA BUS 4', price_idr: 5210400 },
  { name: 'VISA BUS 5', price_idr: 4524000 },
  { name: 'VISA BUS 6', price_idr: 4071600 },
  { name: 'VISA BUS 7', price_idr: 3744000 },
  { name: 'VISA BUS 8', price_idr: 3494400 },
  { name: 'VISA BUS 9', price_idr: 3307200 },
  { name: 'VISA BUS 10', price_idr: 3166800 },
  { name: 'VISA BUS 11', price_idr: 3042000 },
  { name: 'VISA BUS 12', price_idr: 2932800 },
  { name: 'VISA BUS 13', price_idr: 2839200 },
  { name: 'VISA BUS 14', price_idr: 2776800 },
  { name: 'VISA BUS 15', price_idr: 2698800 },
  { name: 'VISA BUS 16', price_idr: 2652000 },
  { name: 'VISA BUS 17', price_idr: 2589600 },
  { name: 'VISA BUS 18', price_idr: 2558400 },
  { name: 'VISA BUS 19', price_idr: 2511600 },
  { name: 'VISA BUS 20', price_idr: 2480400 },
  { name: 'VISA BUS 21', price_idr: 2449200 },
  { name: 'VISA BUS 22', price_idr: 2418000 },
  { name: 'VISA BUS 23', price_idr: 2386800 },
  { name: 'VISA BUS 24', price_idr: 2355600 },
  { name: 'VISA BUS 25', price_idr: 2340000 },
  { name: 'VISA BUS 26', price_idr: 2324400 },
  { name: 'VISA BUS 27', price_idr: 2293200 },
  { name: 'VISA BUS 28', price_idr: 2277600 },
  { name: 'VISA BUS 29', price_idr: 2262000 },
  { name: 'VISA BUS 30', price_idr: 2246400 },
  { name: 'VISA BUS 31', price_idr: 2230800 },
  { name: 'VISA BUS 32', price_idr: 2215200 },
  { name: 'VISA BUS 33', price_idr: 2215200 },
  { name: 'VISA BUS 34', price_idr: 2199600 },
  { name: 'VISA BUS 35', price_idr: 2184000 },
  { name: 'VISA BUS + THAIF 1', price_idr: 20888400 },
  { name: 'VISA BUS + THAIF 2', price_idr: 11341200 },
  { name: 'VISA BUS + THAIF 3', price_idr: 8143200 },
  { name: 'VISA BUS + THAIF 4', price_idr: 6552000 },
  { name: 'VISA BUS + THAIF 5', price_idr: 5600400 },
  { name: 'VISA BUS + THAIF 6', price_idr: 4960800 },
  { name: 'VISA BUS + THAIF 7', price_idr: 4508400 },
  { name: 'VISA BUS + THAIF 8', price_idr: 4165200 },
  { name: 'VISA BUS + THAIF 9', price_idr: 3900000 },
  { name: 'VISA BUS + THAIF 10', price_idr: 3697200 },
  { name: 'VISA BUS + THAIF 11', price_idr: 3510000 },
  { name: 'VISA BUS + THAIF 12', price_idr: 3369600 },
  { name: 'VISA BUS + THAIF 13', price_idr: 3244800 },
  { name: 'VISA BUS + THAIF 14', price_idr: 3151200 },
  { name: 'VISA BUS + THAIF 15', price_idr: 3057600 },
  { name: 'VISA BUS + THAIF 16', price_idr: 2979600 },
  { name: 'VISA BUS + THAIF 17', price_idr: 2901600 },
  { name: 'VISA BUS + THAIF 18', price_idr: 2839200 },
  { name: 'VISA BUS + THAIF 19', price_idr: 2776800 },
  { name: 'VISA BUS + THAIF 20', price_idr: 2730000 },
  { name: 'VISA BUS + THAIF 21', price_idr: 2683200 },
  { name: 'VISA BUS + THAIF 22', price_idr: 2652000 },
  { name: 'VISA BUS + THAIF 23', price_idr: 2605200 },
  { name: 'VISA BUS + THAIF 24', price_idr: 2574000 },
  { name: 'VISA BUS + THAIF 25', price_idr: 2542800 },
  { name: 'VISA BUS + THAIF 26', price_idr: 2511600 },
  { name: 'VISA BUS + THAIF 27', price_idr: 2480400 },
  { name: 'VISA BUS + THAIF 28', price_idr: 2464800 },
];

const transportations = [
  { name: 'TIDAK', price_idr: 0 },
  { name: 'HIACE JED MED', price_idr: 2795000 },
  { name: 'HIACE MED ZIARAH', price_idr: 1290000 },
  { name: 'HIACE MED MAK', price_idr: 2795000 },
  { name: 'HIACE MAK ZIARAH', price_idr: 1720000 },
  { name: 'HIACE MAK JED', price_idr: 1935000 },
  { name: 'HIACE THAIF', price_idr: 3440000 },
  { name: 'GMC JED MED', price_idr: 5160000 },
  { name: 'GMC MED ZIARAH', price_idr: 2042500 },
  { name: 'GMC MED MAK', price_idr: 5160000 },
  { name: 'GMC MAK ZIARAH', price_idr: 2257500 },
  { name: 'GMC MAK JED', price_idr: 2472500 },
  { name: 'GMC THAIF', price_idr: 4085000 },
  { name: 'SEDAN JED MED', price_idr: 2150000 },
  { name: 'SEDAN MED ZIARAH', price_idr: 1075000 },
  { name: 'SEDAN MED MAK', price_idr: 2150000 },
  { name: 'SEDAN MAK ZIARAH', price_idr: 1612500 },
  { name: 'SEDAN MAK JED', price_idr: 1397500 },
  { name: 'SEDAN THAIF', price_idr: 2064000 },
  { name: 'HIACE 5X', price_idr: 9245000 },
];

const muthawif = [
  { name: 'TIDAK', price_idr: 0 },
  { name: 'UMROH SAJA', price_idr: 1075000 },
  { name: '2 HARI', price_idr: 2150000 },
  { name: '3 HARI', price_idr: 3225000 },
  { name: '4 HARI', price_idr: 4300000 },
  { name: '5 HARI', price_idr: 5375000 },
  { name: '6 HARI', price_idr: 6450000 },
  { name: '7 HARI', price_idr: 7525000 },
  { name: '8 HARI', price_idr: 8600000 },
  { name: '9 HARI', price_idr: 9675000 },
  { name: '10 HARI', price_idr: 10750000 },
  { name: '11 HARI', price_idr: 11825000 },
  { name: '12 HARI', price_idr: 12900000 },
  { name: '13 HARI', price_idr: 13975000 },
  { name: '14 HARI', price_idr: 15050000 },
  { name: '15 HARI', price_idr: 16125000 },
  { name: '16 HARI', price_idr: 17200000 },
];

const handlingSaudi = [
  { name: 'TIDAK', price_idr: 0 },
  { name: 'ONLY 4-7', price_idr: 3837600 },
  { name: 'ONLY 8-11', price_idr: 2074800 },
  { name: 'ONLY 12-15', price_idr: 1482000 },
  { name: 'ONLY 16-19', price_idr: 1185600 },
  { name: 'ONLY 20-23', price_idr: 1014000 },
  { name: 'ONLY 24-27', price_idr: 936000 },
  { name: 'ONLY 30', price_idr: 858000 },
  { name: 'FULL 4-7', price_idr: 5928000 },
  { name: 'FULL 8-11', price_idr: 3120000 },
  { name: 'FULL 12-15', price_idr: 2184000 },
  { name: 'FULL 16-19', price_idr: 1716000 },
  { name: 'FULL 20-23', price_idr: 1435200 },
  { name: 'FULL 24-27', price_idr: 1248000 },
  { name: 'FULL 30', price_idr: 1092000 },
  { name: 'BANDARA & DAR', price_idr: 1548000 },
  { name: 'SNACK', price_idr: 129000 },
  { name: 'TIP', price_idr: 1720000 },
  { name: 'TIP & SNACK PRIVATE', price_idr: 1849000 },
];

const handlingDomestik = [
  { name: 'TIDAK', price_idr: 0 },
  { name: 'NASI BOX & BAGASI', price_idr: 250000 },
  { name: 'ZUKAVIA & BAGASI', price_idr: 350000 },
  { name: 'BAGASI ONLY', price_idr: 150000 },
];

const siskopatuh = [
  { name: 'TIDAK', price_idr: 0 },
  { name: 'SISKOPATUH & ASURANSI', price_idr: 500000 },
];

const equipments = [
  { name: 'TIDAK', price_idr: 0 },
  { name: 'AKSESORIS', price_idr: 350000 },
  { name: 'FULL PAKET KOPER', price_idr: 1000000 },
];

const tourPlus = [
  { name: 'TIDAK', price_idr: 0 },
  { name: '1D MESIR', price_idr: 2262000 },
  { name: '2D1N MESIR *3', price_idr: 3276000 },
  { name: '2D1N MESIR *4', price_idr: 3510000 },
  { name: '2D1N MESIR *5', price_idr: 3978000 },
  { name: '3D2N MESIR *3', price_idr: 3588000 },
  { name: '3D2N MESIR *4', price_idr: 3978000 },
  { name: '3D2N MESIR *5', price_idr: 4992000 },
  { name: '4D3N MESIR *3', price_idr: 4056000 },
  { name: '4D3N MESIR *4', price_idr: 4680000 },
  { name: '4D3N MESIR *5', price_idr: 5928000 },
  { name: '5D3N MESIR *3', price_idr: 4680000 },
  { name: '5D3N MESIR *4', price_idr: 5928000 },
  { name: '5D3N MESIR *5', price_idr: 6864000 },
];

const manasik = [
  { name: 'TIDAK', price_idr: 0 },
  { name: 'MANASIK ONLINE', price_idr: 150000 },
  { name: 'MANASIK HOTEL', price_idr: 250000 },
];

const exchangeRates = [
  { name: 'USD', rate_idr: 15000 },
  { name: 'SAR', rate_idr: 4000 },
];

const contact = [
  { name: 'whatsapp', value: '6283133344897' },
  { name: 'email', value: 'syafarmedia@gmail.com' },
  { name: 'instagram', value: '' },
  { name: 'facebook', value: '' },
  { name: 'youtube', value: '' },
  { name: 'twitter', value: '' },
  { name: 'tiktok', value: '' },
  { name: 'linkedin', value: '' },
  { name: 'telegram', value: '' },
  { name: 'line', value: '' },
  { name: 'website', value: '' },
];

const faq = [
  {
    question: 'Apa itu Syafar Tour dan bagaimana cara kerjanya?',
    answer:
      'Syafar Tour adalah agen perjalanan umroh dan haji yang berkomitmen untuk menyediakan layanan terbaik bagi setiap jamaah. Dengan pengalaman bertahun-tahun, kami menawarkan berbagai paket umroh dan haji yang dapat disesuaikan dengan kebutuhan Anda. Didukung oleh tim profesional, kami memastikan kenyamanan, keamanan, dan kemudahan perjalanan anda.',
  },
  {
    question: 'Apa saja paket umroh yang tersedia di Syafar Tour?',
    answer:
      'lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur.',
  },
  {
    question: 'Apa saja yang termasuk dalam paket umroh?',
    answer:
      'lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur.',
  },
  {
    question: 'Bagaimana cara mendaftar umroh di Syafar Tour?',
    answer:
      'lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur.',
  },
  {
    question: 'Apakah saya bisa membayar secara cicilan?',
    answer:
      'lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur.',
  },
];

type ContentType = 'TEXT' | 'IMAGE' | 'BUTTON' | 'BENEFIT_CARD';
const content = [
  {
    page: 'global',
    section: 'header',
    type: 'IMAGE',
    key: 'logo',
    value: {
      src: '/public/content/logo.png',
    },
  },
  {
    page: 'global',
    section: 'footer',
    type: 'TEXT',
    key: 'description',
    value: {
      text: 'Syafar Tour adalah penyedia layanan umroh dan haji terpercaya yang berkomitmen untuk memberikan pengalaman ibadah yang nyaman dan aman. Kami menawarkan berbagai pilihan paket yang dapat disesuaikan dengan kebutuhan Anda.',
    },
  },
  {
    page: 'global',
    section: 'footer',
    type: 'TEXT',
    key: 'copyright',
    value: {
      text: 'Copyright © 2024 Syafar Tour, MMG',
    },
  },
  {
    page: 'home',
    section: 'hero',
    type: 'IMAGE',
    key: 'image',
    value: {
      src: '/public/content/hero.png',
    },
  },
  {
    page: 'home',
    section: 'hero',
    type: 'TEXT',
    key: 'tagline',
    value: {
      text: 'No. 1 di Indonesia',
    },
  },
  {
    page: 'home',
    section: 'hero',
    type: 'TEXT',
    key: 'title',
    value: {
      text: 'Paket Umroh Terbaik untuk Setiap Perjalanan Ibadah Anda',
    },
  },
  {
    page: 'home',
    section: 'hero',
    type: 'TEXT',
    key: 'description',
    value: {
      text: 'Temukan paket reguler, private, atau buat paket umroh sesuai kebutuhan Anda. Perjalanan spiritual lebih mudah dan nyaman dengan kami.',
    },
  },
  {
    page: 'home',
    section: 'hero',
    type: 'BUTTON',
    key: 'button',
    value: {
      text: 'Lihat Produk',
      url: '/produk',
    },
  },
  {
    page: 'home',
    section: 'banner',
    type: 'IMAGE',
    key: 'image 1',
    value: {
      src: '/public/content/banner-1.png',
    },
  },
  {
    page: 'home',
    section: 'banner',
    type: 'IMAGE',
    key: 'image 2',
    value: {
      src: '/public/content/banner-2.png',
    },
  },
  {
    page: 'home',
    section: 'banner',
    type: 'IMAGE',
    key: 'image 3',
    value: {
      src: '/public/content/banner-3.png',
    },
  },
  {
    page: 'home',
    section: 'banner',
    type: 'IMAGE',
    key: 'image 4',
    value: {
      src: '/public/content/banner-4.png',
    },
  },
  {
    page: 'home',
    section: 'banner',
    type: 'IMAGE',
    key: 'image 5',
    value: {
      src: '/public/content/banner-5.png',
    },
  },
  {
    page: 'home',
    section: 'product',
    type: 'TEXT',
    key: 'title',
    value: {
      text: 'Temukan Paket Umroh, Visa, dan Hotel Terbaik untuk Perjalanan Spiritual Anda',
    },
  },
  {
    page: 'home',
    section: 'product',
    type: 'TEXT',
    key: 'subtitle',
    value: {
      text: 'Berbagai pilihan layanan umroh dengan fasilitas lengkap dan fleksibel, siap memenuhi kebutuhan ibadah Anda.',
    },
  },
  {
    page: 'home',
    section: 'about',
    type: 'IMAGE',
    key: 'image',
    value: {
      src: '/public/content/about.png',
    },
  },
  {
    page: 'home',
    section: 'about',
    type: 'TEXT',
    key: 'title',
    value: {
      text: 'Tentang',
    },
  },
  {
    page: 'home',
    section: 'about',
    type: 'TEXT',
    key: 'subtitle',
    value: {
      text: 'Kenali Syafar Tour Lebih Dekat',
    },
  },
  {
    page: 'home',
    section: 'about',
    type: 'TEXT',
    key: 'description',
    value: {
      text: 'Syafar Tour adalah agen perjalanan umroh dan haji yang berkomitmen untuk menyediakan layanan terbaik bagi setiap jamaah. Dengan pengalaman bertahun-tahun, kami menawarkan berbagai paket umroh dan haji yang dapat disesuaikan dengan kebutuhan Anda. Didukung oleh tim profesional, kami memastikan kenyamanan, keamanan, dan kemudahan perjalanan anda.',
    },
  },
  {
    page: 'home',
    section: 'benefit',
    type: 'TEXT',
    key: 'title',
    value: {
      text: 'Mengapa memilih Syafar Tour?',
    },
  },
  {
    page: 'home',
    section: 'benefit',
    type: 'TEXT',
    key: 'subtitle',
    value: {
      text: 'Layanan Terbaik untuk Perjalanan Ibadah yang Aman dan Nyaman',
    },
  },
  {
    page: 'home',
    section: 'benefit',
    type: 'BENEFIT_CARD',
    key: 'card 1',
    value: {
      imageSrc: '/public/content/benefit-1.png',
      title: 'Layanan Profesional',
      description: 'Tim berpengalaman siap memberikan bantuan dan informasi untuk memastikan kenyamanan Anda.',
    },
  },
  {
    page: 'home',
    section: 'benefit',
    type: 'BENEFIT_CARD',
    key: 'card 2',
    value: {
      imageSrc: '/public/content/benefit-2.png',
      title: 'Fasilitas Lengkap',
      description:
        'Pilih akomodasi sesuai preferensi Anda, dari hotel bintang 3 hingga bintang 5, dengan layanan terbaik.',
    },
  },
  {
    page: 'home',
    section: 'benefit',
    type: 'BENEFIT_CARD',
    key: 'card 3',
    value: {
      imageSrc: '/public/content/benefit-3.png',
      title: 'Harga Terbaik',
      description:
        'Kami menawarkan beragam paket umroh, dari reguler hingga private, serta opsi custom untuk kebutuhan khusus.',
    },
  },
  {
    page: 'home',
    section: 'faq',
    type: 'TEXT',
    key: 'title',
    value: {
      text: 'Pertanyaan yang Sering Diajukan',
    },
  },
  {
    page: 'home',
    section: 'faq',
    type: 'TEXT',
    key: 'subtitle',
    value: {
      text: 'Semua yang Perlu Anda Ketahui Sebelum Memulai Perjalanan Ibadah',
    },
  },
  {
    page: 'home',
    section: 'cta',
    type: 'IMAGE',
    key: 'image',
    value: {
      src: '/public/content/packages.png',
    },
  },
  {
    page: 'home',
    section: 'cta',
    type: 'TEXT',
    key: 'title',
    value: {
      text: 'Mulai Perjalanan Ibadah Anda Sekarang!',
    },
  },
  {
    page: 'home',
    section: 'cta',
    type: 'TEXT',
    key: 'subtitle',
    value: {
      text: 'Nikmati layanan terbaik dengan paket fleksibel, fasilitas lengkap, dan bimbingan dari tim profesional kami.',
    },
  },
  {
    page: 'home',
    section: 'cta',
    type: 'BUTTON',
    key: 'button',
    value: {
      text: 'Lihat Paket',
      url: '/paket',
    },
  },
  {
    page: 'paket',
    section: 'hero',
    type: 'TEXT',
    key: 'title',
    value: {
      text: 'Buat Paket Umroh yang Sesuai dengan Kebutuhan Anda',
    },
  },
  {
    page: 'paket',
    section: 'hero',
    type: 'TEXT',
    key: 'description',
    value: {
      text: 'Paket Custom Syafar Tour memberi Anda fleksibilitas penuh untuk merencanakan perjalanan umroh sesuai keinginan. Pilih sendiri layanan yang Anda butuhkan, mulai dari visa, hotel, transportasi, hingga tiket pesawat. Paket ini dirancang khusus bagi Anda yang menginginkan pengalaman umroh yang lebih personal dan sesuai dengan preferensi.',
    },
  },
  {
    page: 'paket',
    section: 'hero',
    type: 'TEXT',
    key: 'button_text',
    value: {
      text: 'Buat Paket Custom Sekarang',
    },
  },
  {
    page: 'registrasi-mitra',
    section: 'bank',
    type: 'IMAGE',
    key: 'image',
    value: {
      src: '/public/content/bank.png',
    },
  },
  {
    page: 'registrasi-mitra',
    section: 'bank',
    type: 'TEXT',
    key: 'name',
    value: {
      text: 'Bank BCA',
    },
  },
  {
    page: 'registrasi-mitra',
    section: 'bank',
    type: 'TEXT',
    key: 'account_number',
    value: {
      text: '1234567890',
    },
  },
  {
    page: 'registrasi-mitra',
    section: 'bank',
    type: 'TEXT',
    key: 'account_name',
    value: {
      text: 'Syafar Media',
    },
  },
];

const menuItems = [
  { name: 'Home', url: '/' },
  { name: 'Produk', url: '/produk' },
  { name: 'Paket', url: '/paket' },
  { name: 'FAQ', url: '/faq' },
];

type FooterType = 'MENU' | 'LINK' | 'SOCIAL_MEDIA';
const footerItems = [
  { name: 'Tautan', type: 'MENU', order: 1 },
  {
    name: 'Kebijakan',
    type: 'LINK',
    order: 2,
    items: [
      { name: 'Kebijakan Privasi', url: '/kebijakan-privasi' },
      { name: 'Syarat dan Ketentuan', url: '/syarat-dan-ketentuan' },
      { name: 'Kebijakan Pengembalian Dana', url: '/kebijakan-pengembalian-dana' },
    ],
  },
  {
    name: 'Social Media',
    type: 'SOCIAL_MEDIA',
    order: 3,
    items: [
      { name: 'Facebook', url: 'https://www.facebook.com/syafartour/' },
      { name: 'Instagram', url: 'https://www.instagram.com/syafartour/' },
      { name: 'Twitter', url: 'https://twitter.com/syafartour/' },
      { name: 'Youtube', url: 'https://www.youtube.com/syafartour/' },
      { name: 'Telegram', url: 'https://t.me/syafartour/' },
      { name: 'LinkedIn', url: 'https://www.linkedin.com/syafartour/' },
      { name: 'Line', url: 'https://line.me/syafartour/' },
      { name: 'TikTok', url: 'https://www.tiktok.com/syafartour/' },
    ],
  },
];

const colors = [
  {
    name: 'primary',
    hex: '#D7AF57',
  },
  {
    name: 'primary-dark',
    hex: '#B48E3F',
  },
  {
    name: 'secondary',
    hex: '#353535',
  },
  {
    name: 'dark',
    hex: '#272727',
  },
  {
    name: 'light',
    hex: '#FDFFFC',
  },
];

async function createUsers() {
  for (const user of users) {
    const existingUser = await db.queryOne<{ id: string }>('SELECT id FROM user WHERE email = ?', [user.email]);

    const hashedPassword = await bcrypt.hash(user.password, 10);

    if (!existingUser) {
      await db.query(
        `INSERT INTO user (id, email, full_name, whatsapp_number, password, role) 
           VALUES (?, ?, ?, ?, ?, ?)`,
        [uuidv4(), user.email, user.full_name, user.whatsapp_number, hashedPassword, user.role as Role]
      );
    } else {
      console.log(`User '${user.email}' already exists`);
    }
  }
  console.log('Users seeded successfully!');
}

async function seedPackageItems() {
  for (const packageItem of packageTypes) {
    const packageItemId = uuidv4();

    await db.query(`INSERT INTO package_type (id, name) VALUES (?, ?)`, [packageItemId, packageItem.name]);

    const itemKey = packageItem.name.replace(/\s/g, '') as PackageTypes;
    const relatedItems = itemData[itemKey];

    for (const item of relatedItems) {
      await db.query(
        `INSERT INTO user_package_option (id, name, price, order_number, package_type_id) 
           VALUES (?, ?, ?, ?, ?)`,
        [uuidv4(), item.name, item.price, item.order_number ?? 0, packageItemId]
      );
    }
  }

  console.log('Package items seeded successfully!');
}

async function createCategories() {
  for (const category of CategoryData) {
    const existingCategory = await db.queryOne<{ id: string }>('SELECT id FROM category WHERE name = ? LIMIT 1', [
      category.name,
    ]);

    if (!existingCategory) {
      await db.query(`INSERT INTO category (id, name) VALUES (?, ?)`, [uuidv4(), category.name]);
    } else {
      console.log(`Category '${category.name}' already exists`);
    }
  }
  console.log('Categories seeded successfully!');
}

async function createPeriods() {
  for (const period of periods) {
    const existingPeriod = await db.queryOne<{ id: string }>(
      'SELECT id FROM period WHERE start_date = ? AND end_date = ?',
      [period.start_date, period.end_date]
    );

    if (!existingPeriod) {
      await db.query(`INSERT INTO period (id, category, start_date, end_date) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        period.category as PeriodCategory,
        period.start_date,
        period.end_date,
      ]);
    } else {
      console.log(`Period from ${period.start_date} to ${period.end_date} already exists`);
    }
  }
  console.log('Periods seeded successfully!');
}

async function createVendors() {
  for (const vendor of vendors) {
    const existingVendor = await db.queryOne<{ id: string }>('SELECT id FROM vendor WHERE name = ? LIMIT 1', [
      vendor.name,
    ]);

    if (!existingVendor) {
      await db.query(`INSERT INTO vendor (id, name) VALUES (?, ?)`, [uuidv4(), vendor.name]);
    } else {
      console.log(`Vendor '${vendor.name}' already exists`);
    }
  }
  console.log('Vendors seeded successfully!');
}

async function seedHotels() {
  for (const vendorGroup of hotels) {
    const vendor = await db.queryOne<{ id: string }>('SELECT id FROM vendor WHERE name = ? LIMIT 1', [
      vendorGroup.vendor,
    ]);

    if (!vendor) {
      console.error(`Vendor ${vendorGroup.vendor} not found. Skipping...`);
      continue;
    }

    for (const hotel of vendorGroup.hotels) {
      const newHotelId = uuidv4();
      await db.query(`INSERT INTO hotel (id, vendor_id, name, city, order_number) VALUES (?, ?, ?, ?, ?)`, [
        newHotelId,
        vendor.id,
        hotel.name,
        hotel.city as City,
        0,
      ]);

      const periods = await db.query<{ id: string }>('SELECT id FROM period');

      const periodPrices = periods.map((period) => [
        uuidv4(),
        newHotelId,
        period.id,
        hotel.prices.double,
        hotel.prices.triple,
        hotel.prices.quad,
      ]);

      await db.query(
        `INSERT INTO hotel_period_price (id, hotel_id, period_id, price_double, price_triple, price_quad) VALUES ?`,
        [periodPrices]
      );
    }
  }
  console.log('Hotels seeded successfully!');
}

async function createAirlines() {
  for (const airline of airlines) {
    const existingAirline = await db.queryOne<{ id: string }>('SELECT id FROM airline WHERE name = ? LIMIT 1', [
      airline.name,
    ]);

    if (!existingAirline) {
      await db.query(`INSERT INTO airline (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        airline.name,
        airline.price,
        0,
      ]);
    } else {
      console.log(`Airline '${airline.name}' already exists`);
    }
  }
  console.log('Airlines seeded successfully!');
}

async function seedVisa() {
  for (const visaItem of visa) {
    const existingVisa = await db.queryOne<{ id: string }>('SELECT id FROM visa WHERE name = ? LIMIT 1', [
      visaItem.name,
    ]);

    if (!existingVisa) {
      await db.query(`INSERT INTO visa (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        visaItem.name,
        visaItem.price_idr,
        0,
      ]);
    } else {
      console.log(`Visa '${visaItem.name}' already exists`);
    }
  }
  console.log('Visa seeded successfully!');
}

async function seedTransportations() {
  for (const transportation of transportations) {
    const existingTransportation = await db.queryOne<{ id: string }>(
      'SELECT id FROM transportation WHERE name = ? LIMIT 1',
      [transportation.name]
    );

    if (!existingTransportation) {
      await db.query(`INSERT INTO transportation (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        transportation.name,
        transportation.price_idr,
        0,
      ]);
    } else {
      console.log(`Transportation '${transportation.name}' already exists`);
    }
  }
  console.log('Transportations seeded successfully!');
}

async function seedMuthawwif() {
  for (const muthawwifItem of muthawif) {
    const existingMuthawwif = await db.queryOne<{ id: string }>('SELECT id FROM muthawif WHERE name = ? LIMIT 1', [
      muthawwifItem.name,
    ]);

    if (!existingMuthawwif) {
      await db.query(`INSERT INTO muthawif (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        muthawwifItem.name,
        muthawwifItem.price_idr,
        0,
      ]);
    } else {
      console.log(`Muthawwif '${muthawwifItem.name}' already exists`);
    }
  }
  console.log('Muthawwif seeded successfully!');
}

async function seedHandlingSaudi() {
  for (const handlingItem of handlingSaudi) {
    const existingHandling = await db.queryOne<{ id: string }>('SELECT id FROM handling_saudi WHERE name = ? LIMIT 1', [
      handlingItem.name,
    ]);

    if (!existingHandling) {
      await db.query(`INSERT INTO handling_saudi (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        handlingItem.name,
        handlingItem.price_idr,
        0,
      ]);
      console.log(`Handling '${handlingItem.name}' added successfully`);
    } else {
      console.log(`Handling '${handlingItem.name}' already exists`);
    }
  }
  console.log('Handling Saudi seeded successfully!');
}

async function seedHandlingDomestik() {
  for (const handlingItem of handlingDomestik) {
    const existingHandling = await db.queryOne<{ id: string }>(
      'SELECT id FROM handling_domestic WHERE name = ? LIMIT 1',
      [handlingItem.name]
    );

    if (!existingHandling) {
      await db.query(`INSERT INTO handling_domestic (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        handlingItem.name,
        handlingItem.price_idr,
        0,
      ]);
    } else {
      console.log(`Handling '${handlingItem.name}' already exists`);
    }
  }
  console.log('Handling Domestik seeded successfully!');
}

async function seedSiskopatuh() {
  for (const siskopatuhItem of siskopatuh) {
    const existingSiskopatuh = await db.queryOne<{ id: string }>('SELECT id FROM siskopatuh WHERE name = ? LIMIT 1', [
      siskopatuhItem.name,
    ]);

    if (!existingSiskopatuh) {
      await db.query(`INSERT INTO siskopatuh (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        siskopatuhItem.name,
        siskopatuhItem.price_idr,
        0,
      ]);
    } else {
      console.log(`Siskopatuh '${siskopatuhItem.name}' already exists`);
    }
  }
  console.log('Siskopatuh seeded successfully!');
}

async function seedEquipments() {
  for (const equipment of equipments) {
    const existingEquipment = await db.queryOne<{ id: string }>('SELECT id FROM equipment WHERE name = ? LIMIT 1', [
      equipment.name,
    ]);

    if (!existingEquipment) {
      await db.query(`INSERT INTO equipment (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        equipment.name,
        equipment.price_idr,
        0,
      ]);
    } else {
      console.log(`Equipment '${equipment.name}' already exists`);
    }
  }
  console.log('Equipments seeded successfully!');
}

async function seedTourPlus() {
  for (const tourPlusItem of tourPlus) {
    const existingTourPlus = await db.queryOne<{ id: string }>('SELECT id FROM tour_plus WHERE name = ? LIMIT 1', [
      tourPlusItem.name,
    ]);

    if (!existingTourPlus) {
      await db.query(`INSERT INTO tour_plus (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        tourPlusItem.name,
        tourPlusItem.price_idr,
        0,
      ]);
    } else {
      console.log(`Tour Plus '${tourPlusItem.name}' already exists`);
    }
  }
  console.log('Tour Plus seeded successfully!');
}

async function seedManasik() {
  for (const manasikItem of manasik) {
    const existingManasik = await db.queryOne<{ id: string }>('SELECT id FROM manasik WHERE name = ? LIMIT 1', [
      manasikItem.name,
    ]);

    if (!existingManasik) {
      await db.query(`INSERT INTO manasik (id, name, price_idr, order_number) VALUES (?, ?, ?, ?)`, [
        uuidv4(),
        manasikItem.name,
        manasikItem.price_idr,
        0,
      ]);
    } else {
      console.log(`Manasik '${manasikItem.name}' already exists`);
    }
  }
  console.log('Manasik seeded successfully!');
}

async function seedExchangeRates() {
  for (const exchangeRate of exchangeRates) {
    const existingExchangeRate = await db.queryOne<{ id: string }>(
      'SELECT id FROM exchange_rate WHERE currency = ? LIMIT 1',
      [exchangeRate.name]
    );

    if (!existingExchangeRate) {
      await db.query(`INSERT INTO exchange_rate (id, currency, rate_idr) VALUES (?, ?, ?)`, [
        uuidv4(),
        exchangeRate.name,
        exchangeRate.rate_idr,
      ]);
    } else {
      console.log(`Exchange Rate '${exchangeRate.name}' already exists`);
    }
  }
  console.log('Exchange Rates seeded successfully!');
}

async function seedContact() {
  for (const contactItem of contact) {
    const existingContact = await db.queryOne<{ id: string }>('SELECT id FROM contact WHERE name = ? LIMIT 1', [
      contactItem.name,
    ]);

    if (!existingContact) {
      await db.query(`INSERT INTO contact (id, name, value) VALUES (?, ?, ?)`, [
        uuidv4(),
        contactItem.name,
        contactItem.value,
      ]);
    } else {
      console.log(`Contact '${contactItem.name}' already exists`);
    }
  }
  console.log('Contact seeded successfully!');
}

async function seedFaq() {
  for (const faqItem of faq) {
    const existingFaq = await db.queryOne<{ id: string }>('SELECT id FROM faq WHERE question = ? LIMIT 1', [
      faqItem.question,
    ]);

    if (!existingFaq) {
      await db.query(`INSERT INTO faq (id, question, answer) VALUES (?, ?, ?)`, [
        uuidv4(),
        faqItem.question,
        faqItem.answer,
      ]);
    } else {
      console.log(`FAQ '${faqItem.question}' already exists`);
    }
  }
  console.log('FAQs seeded successfully!');
}

async function seedContent() {
  for (const contentItem of content) {
    const existingContent = await db.queryOne<{ id: string }>(
      'SELECT id FROM content WHERE page = ? AND section = ? AND `key` = ? LIMIT 1',
      [contentItem.page, contentItem.section, contentItem.key]
    );

    const contentId = uuidv4();

    if (!existingContent) {
      await db.query(`INSERT INTO content (id, page, section, type, \`key\`, value) VALUES (?, ?, ?, ?, ?, ?)`, [
        contentId,
        contentItem.page,
        contentItem.section,
        contentItem.type as ContentType,
        contentItem.key,
        JSON.stringify(contentItem.value),
      ]);
    } else {
      console.log(`Content '${contentItem.key}' already exists`);
    }
  }
  console.log('Content seeded successfully!');
}

async function seedMenuItems() {
  for (const menuItem of menuItems) {
    const existingMenuItem = await db.queryOne<{ id: string }>('SELECT id FROM menu_item WHERE name = ? LIMIT 1', [
      menuItem.name,
    ]);

    if (!existingMenuItem) {
      await db.query(`INSERT INTO menu_item (id, name, url) VALUES (?, ?, ?)`, [uuidv4(), menuItem.name, menuItem.url]);
    } else {
      console.log(`Menu Item '${menuItem.name}' already exists`);
    }
  }
  console.log('Menu Items seeded successfully!');
}

async function seedFooterItems() {
  for (const footerItem of footerItems) {
    const existingFooter = await db.queryOne<{ id: string }>('SELECT id FROM footer WHERE name = ? LIMIT 1', [
      footerItem.name,
    ]);

    if (!existingFooter) {
      const footerId = uuidv4();
      await db.query(`INSERT INTO footer (id, name, type, \`order\`) VALUES (?, ?, ?, ?)`, [
        footerId,
        footerItem.name,
        footerItem.type as FooterType,
        footerItem.order,
      ]);

      if (footerItem.items) {
        const footerItemsData = footerItem.items.map((item) => [uuidv4(), item.name, item.url, footerId]);

        await db.query(`INSERT INTO footer_item (id, name, url, footer_id) VALUES ?`, [footerItemsData]);
      }
    } else {
      console.log(`Footer Item '${footerItem.name}' already exists`);
    }
  }
  console.log('Footer Items seeded successfully!');
}

async function seedColors() {
  for (const color of colors) {
    const existingColor = await db.queryOne<{ id: string }>('SELECT id FROM color WHERE name = ? LIMIT 1', [
      color.name,
    ]);

    if (!existingColor) {
      await db.query(`INSERT INTO color (id, name, hex) VALUES (?, ?, ?)`, [uuidv4(), color.name, color.hex]);
    } else {
      console.log(`Color '${color.name}' already exists`);
    }
  }
  console.log('Colors seeded successfully!');
}

async function main() {
  try {
    console.log('Starting seeding process...');
    // await createUsers();
    // await seedPackageItems();
    // await createCategories();
    // await createPeriods();
    // await createVendors();
    // await seedHotels();
    // await createAirlines();
    // await seedVisa();
    // await seedTransportations();
    // await seedMuthawwif();
    // await seedHandlingSaudi();
    // await seedHandlingDomestik();
    // await seedSiskopatuh();
    // await seedEquipments();
    // await seedTourPlus();
    // await seedManasik();
    // await seedExchangeRates();
    // await seedContact();
    // await seedFaq();
    await seedContent();
    // await seedMenuItems();
    // await seedFooterItems();
    // await seedColors();
    console.log('Seeding completed successfully.');
  } catch (error) {
    console.error('Error during seeding:', error);
  } finally {
    await db.close();
  }
}

main().catch((e) => {
  console.error(e);
  process.exit(1);
});
